﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;

namespace Framework.Log
{
    /* Memo : ロガーについて
     * このロガーは汎用的に作ってあり、アプリケーションに特化はしていない。
     * そのため、アプリケーションに特化したメソッドは拡張メソッドにより追加することになる。
     * たとえば、
     * public static class ILoggerExtension
     * {
     *     public static void ErrorWithID(this ILogger target, string message)
     *     {
     *         var actualMessage = string.Format("user id:{0}\nmessage:{1}", UserMgr.GetLoginUserID(), message);
     *         target.Error(actualMessage);
     *     }
     * }
     * のように記述する。
     */

    /// <summary>
    /// ログ用のインターフェイスです。
    /// デバッグ用の情報を取得するためには、Loggerクラスを使用してください。
    /// </summary>
    public interface ILogger
    {
        /// <summary>
        /// スタックトレース情報を含むメッセージをレベルErrorでログします。
        /// </summary>
        void Error(object message, Exception e);

        /// <summary>
        /// メッセージをレベルErrorでログします。
        /// </summary>
        void Error(object message);

        /// <summary>
        /// フォーマットされたメッセージをレベルErrorでログします。
        /// </summary>
        void ErrorFormat(string format, params object[] args);

        /// <summary>
        /// スタックトレース情報を含むメッセージをレベルFatalでログします。
        /// </summary>
        void Fatal(object message, Exception e);

        /// <summary>
        /// メッセージをレベルFatalでログします。
        /// </summary>
        void Fatal(object message);

        /// <summary>
        /// フォーマットされたメッセージをレベルFatalでログします。
        /// </summary>
        void FatalFormat(string format, params object[] args);

        /// <summary>
        /// スタックトレース情報を含むメッセージをレベルInfoでログします。
        /// </summary>
        void Info(object message, Exception e);

        /// <summary>
        /// メッセージをレベルInfoでログします。
        /// </summary>
        void Info(object message);

        /// <summary>
        /// フォーマットされたメッセージをレベルInfoでログします。
        /// </summary>
        void InfoFormat(string format, params object[] args);

        /// <summary>
        /// スタックトレース情報を含むメッセージをレベルWarnでログします。
        /// </summary>
        void Warn(object message, Exception e);

        /// <summary>
        /// メッセージをレベルWarnでログします。
        /// </summary>
        void Warn(object message);

        /// <summary>
        /// フォーマットされたメッセージをレベルWarnでログします。
        /// </summary>
        void WarnFormat(string format, params object[] args);

        /// <summary>
        /// レベルDebugのメッセージをログする設定になっているかどうかを返します。
        /// </summary>
        bool IsDebugEnabled { get; }

        /// <summary>
        /// レベルErrorのメッセージをログする設定になっているかどうかを返します。
        /// </summary>
        bool IsErrorEnabled { get; }

        /// <summary>
        /// レベルFatalのメッセージをログする設定になっているかどうかを返します。
        /// </summary>
        bool IsFatalEnabled { get; }

        /// <summary>
        /// レベルInfoのメッセージをログする設定になっているかどうかを返します。
        /// </summary>
        bool IsInfoEnabled { get; }

        /// <summary>
        /// レベルWarnのメッセージをログする設定になっているかどうかを返します。
        /// </summary>
        bool IsWarnEnabled { get; }

        /// <summary>
        /// このメソッドは直接使用せずに、拡張メソッドを使用してください。
        /// このメソッドを直接呼び出した場合、Releaseモード時にも余分なメソッドコールが行われてしまい、効率が悪くなります。
        /// </summary>
        [Obsolete]
        void XXX_Debug(object message, Exception e);

        /// <summary>
        /// このメソッドは直接使用せずに、拡張メソッドを使用してください。
        /// このメソッドを直接呼び出した場合、Releaseモード時にも余分なメソッドコールが行われてしまい、効率が悪くなります。
        /// </summary>
        [Obsolete]
        void XXX_Debug(object message);

        /// <summary>
        /// このメソッドは直接使用せずに、拡張メソッドを使用してください。
        /// このメソッドを直接呼び出した場合、Releaseモード時にも余分なメソッドコールが行われてしまい、効率が悪くなります。
        /// </summary>
        [Obsolete]
        void XXX_DebugFormat(string format, params object[] args);
    }

    /* Memo : Conditional属性について
     * 以下の拡張メソッドには、各メソッドにConditional属性が付与されている。
     * この属性により、Debugモードでビルドした場合にのみこれらのメソッド呼び出しが有効となる。
     * 引数の評価もDebug時にしか行われないため、
     * これらの拡張メソッドの引数として時間のかかる処理を直接埋め込んだ場合でも、
     * Releaseビルド時にはこれらのメソッド呼び出し自体が取り除かれるため、何も問題は発生しない。
     * この属性はinterfaceからのメソッドや、virtualのメソッドに対しては使用できないため、
     * 拡張メソッドとして実装している。
     */

// Obsoluteメソッドの警告回避
// このクラス内部で呼び出しているObsoluteメソッドはこのクラス内で使うことを想定されて作成しているため、
// このプラグマ指定に問題はない
#pragma warning disable 612
    public static class ILoggerExtension
    {

        /// <summary>
        /// スタックトレース情報を含むメッセージをレベルDebugでログします。
        /// 引数の評価を含むこのメソッド呼び出しの全体は、Releaseモードでは削除されます。
        /// </summary>
        [Conditional("DEBUG")]
        public static void Debug(this ILogger target, object message, Exception e)
        {
            target.XXX_Debug(message, e);
        }

        /// <summary>
        /// メッセージをレベルDebugでログします。
        /// 引数の評価を含むこのメソッド呼び出しの全体は、Releaseモードでは削除されます。
        /// </summary>
        [Conditional("DEBUG")]
        public static void Debug(this ILogger target, object message)
        {
            target.XXX_Debug(message);
        }

        /// <summary>
        /// フォーマットされたメッセージをレベルDebugでログします。
        /// 引数の評価を含むこのメソッド呼び出しの全体は、Releaseモードでは削除されます。
        /// </summary>
        [Conditional("DEBUG")]
        public static void DebugFormat(this ILogger target, string format, params object[] args)
        {
            target.XXX_DebugFormat(format, args);
        }
    }
// 以降では再びObsoluteメソッド呼び出しに対する警告を報告するように設定
#pragma warning restore 612

}
